<?php

namespace App\Models;


use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Jobpost extends Model
{

    protected $table = 'job_posts';
    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        // Job Details
        'job_title',
        'location',
        'workplace_option',
        'job_type',
        'category',
        'description',

        // Salary & Benefits
        'salary_type',
        'currency',
        'min_salary',
        'max_salary',
        'salary_period',
        'benefits',

        // Company Info
        'company_name',
        'company_website',
        'company_description',
        'company_size',
        'industry',

        // Requirements
        'experience_level',
        'education_level',
        'skills',
        'require_cover_letter',
        'require_portfolio',
        'require_references',

        // Additional Info
        'preview_notes',
        'status',

        // Foreign key
        'user_id',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'benefits' => 'array',
        'require_cover_letter' => 'boolean',
        'require_portfolio' => 'boolean',
        'require_references' => 'boolean',
        'min_salary' => 'decimal:2',
        'max_salary' => 'decimal:2',
    ];

    /**
     * Get the employer that owns the job post.
     */
    public function employer()
    {
        return $this->belongsTo(User::class);
    }

    public function userData(){
    return $this->hasOne(User::class, 'id', 'user_id');
}

   public function questions(): HasMany
{
    return $this->hasMany(JobQuestion::class);
}

   public function businessDetails(){

      return $this->hasOne(Businessprofile::class, 'user_id', 'user_id');
   }
    /**
     * Scope a query to only include published job posts.
     */
    public function scopePublished($query)
    {
        return $query->where('status', 'published');
    }

    /**
     * Scope a query to only include draft job posts.
     */
    public function scopeDraft($query)
    {
        return $query->where('status', 'draft');
    }

    /**
     * Get the formatted salary range.
     */
    public function getFormattedSalaryAttribute(): string
    {
        if ($this->salary_type === 'range' && $this->min_salary && $this->max_salary) {
            $symbol = $this->getCurrencySymbol();
            $period = $this->getSalaryPeriod();

            return "{$symbol}{$this->min_salary} - {$symbol}{$this->max_salary} {$period}";
        }

        if ($this->salary_type === 'fixed' && $this->min_salary) {
            $symbol = $this->getCurrencySymbol();
            $period = $this->getSalaryPeriod();

            return "{$symbol}{$this->min_salary} {$period}";
        }

        return ucfirst($this->salary_type);
    }

    /**
     * Get the currency symbol.
     */
    private function getCurrencySymbol(): string
    {
        $symbols = [
            'USD' => '$',
            'EUR' => '€',
            'GBP' => '£',
            'JPY' => '¥',
        ];

        return $symbols[$this->currency] ?? '$';
    }

    /**
     * Get the salary period text.
     */
    private function getSalaryPeriod(): string
    {
        $periods = [
            'hour' => 'per hour',
            'week' => 'per week',
            'month' => 'per month',
            'year' => 'per year',
        ];

        return $periods[$this->salary_period] ?? 'per month';
    }

    /**
     * Get the workplace option name.
     */
    public function getWorkplaceOptionNameAttribute(): string
    {
        $options = [
            'on-site' => 'On-site',
            'remote' => 'Remote',
            'hybrid' => 'Hybrid',
        ];

        return $options[$this->workplace_option] ?? 'On-site';
    }

    /**
     * Get the job type name.
     */
    public function getJobTypeNameAttribute(): string
    {
        $types = [
            'full-time' => 'Full Time',
            'part-time' => 'Part Time',
            'contract' => 'Contract',
            'temporary' => 'Temporary',
            'internship' => 'Internship',
        ];

        return $types[$this->job_type] ?? 'Full Time';
    }
}
