<?php

namespace App\Livewire\Web\Index\Modal;

use App\Models\User;
use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Str;
use Laravel\Socialite\Facades\Socialite;
use Livewire\Component;


class Signinmodal extends Component
{
    public $email;
    public $password;
    public $showPassword = false;
    public $forgotPasswordMode = false;

    protected $listeners = ['opensigninmodal' => 'openmodal', 'closesigninmodal' => 'closeModal'];
    public $opensigninmodal = false;

    protected $rules = [
        'email' => 'required|email',
        'password' => 'required|min:6',
    ];

    public function submit()
    {
        $this->validate();

        if (Auth::attempt(['email' => $this->email, 'password' => $this->password])) {
            session()->flash('success', 'Welcome back!');
            return redirect()->route('skillpot.home');
        } else {
            $this->addError('email', 'Invalid email or password.');
        }
    }

    public function sendResetLink()
    {
        $this->validate(['email' => 'required|email']);

        $status = Password::sendResetLink(['email' => $this->email]);

        if ($status === Password::RESET_LINK_SENT) {
            session()->flash('success', __($status));
            $this->forgotPasswordMode = false;
        } else {
            session()->flash('error', __($status));
        }
    }

    public function toggleForgotPassword()
    {
        $this->forgotPasswordMode = !$this->forgotPasswordMode;
        $this->resetErrorBag();
        $this->resetValidation();
    }

    public function openmodal()
    {
        $this->opensigninmodal = true;
        $this->forgotPasswordMode = false;
    }

    public function closeModal()
    {
        $this->opensigninmodal = false;
        $this->forgotPasswordMode = false;
        $this->resetErrorBag();
        $this->resetValidation();
        $this->reset(['email', 'password']);
    }

    public function signupmodal()
    {
        $this->dispatch('closesigninmodal');
        $this->dispatch('signupmodalview');
    }

 public function googleSignIn($credential)
    {
        try {
            // Verify the Google token and get user data
            $googleUser = $this->verifyGoogleToken($credential);

            if (!$googleUser) {
                throw new Exception('Invalid Google token');
            }

            // Find user by Google ID or email
            $user = User::where('google_id', $googleUser['sub'])
                        ->orWhere('email', $googleUser['email'])
                        ->first();

            if (!$user) {
                // Create new user
                $user = User::create([
                    'name' => $googleUser['name'],
                    'email' => $googleUser['email'],
                    'google_id' => $googleUser['sub'],
                    'email_verified_at' => now(), // Google verified the email
                    'password' => bcrypt(uniqid()), // Random password
                ]);
            } else {
                // Update existing user with Google ID if not set
                if (!$user->google_id) {
                    $user->google_id = $googleUser['sub'];
                    $user->save();
                }
            }

            // Log the user in
            Auth::login($user);

            // Close the modal
            $this->closeModal();

            // Redirect to dashboard or intended page
            return redirect()->intended('/dashboard');

        } catch (Exception $e) {
            Log::error('Google authentication error: ' . $e->getMessage());
            $this->addError('google_auth', 'Failed to authenticate with Google.');
            session()->flash('error', 'Google authentication failed: ' . $e->getMessage());
        }
    }

    /**
     * Verify Google token and return user data
     */
    private function verifyGoogleToken($credential)
    {
        try {
            // Verify the token using Google's API
            $response = Http::get('https://oauth2.googleapis.com/tokeninfo', [
                'id_token' => $credential
            ]);

            if ($response->successful()) {
                $data = $response->json();

                // Validate the token
                if ($data['aud'] !== config('services.google.client_id')) {
                    throw new Exception('Invalid token audience');
                }

                if ($data['exp'] < time()) {
                    throw new Exception('Token has expired');
                }

                return $data;
            }

            throw new Exception('Token verification failed');

        } catch (Exception $e) {
            Log::error('Google token verification error: ' . $e->getMessage());
            throw new Exception('Token verification failed: ' . $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.web.index.modal.signinmodal');
    }
}
