<?php

namespace App\Livewire\Web\Index\Employer\Job;

use App\Models\Businessprofile;
use App\Models\Jobpost;
use App\Models\JobQuestion;
use App\Models\JobQuestionAnswer;
use App\Services\AIServices;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;

class Createjob extends Component
{
    public $currentTab = 'job-details';
    public $jobId = null; // To track the job post being created

    public $businessname;
    protected $listeners = ['savingjob'];
    public $jobDetails = [
        'jobTitle' => '',
        'location' => '',
        'workplaceOption' => 'on-site',
        'jobType' => '',
        'category' => '',
        'description' => ''
    ];

    public $salaryBenefits = [
        'salaryType' => '',
        'currency' => 'USD',
        'minSalary' => '',
        'maxSalary' => '',
        'salaryPeriod' => 'month',
        'benefits' => []
    ];

    public $companyInfo = [
        'companyWebsite' => '',
        'companySize' => '',
        'industry' => ''
    ];

    public $requirements = [
        'experienceLevel' => '',
        'educationLevel' => '',
        'skills' => '',
        'coverLetter' => false,
        'portfolio' => false,
        'references' => false
    ];

    public $previewNotes = '';

    // Available options
    public $workplaceOptions = [
        'on-site' => ['icon' => 'building', 'name' => 'On-site', 'desc' => 'Work from office'],
        'remote' => ['icon' => 'home', 'name' => 'Remote', 'desc' => 'Work from anywhere'],
        'hybrid' => ['icon' => 'blender-phone', 'name' => 'Hybrid', 'desc' => 'Mix of both']
    ];

    public $benefitsList = [
        'health' => ['icon' => 'stethoscope', 'name' => 'Health Insurance'],
        'retirement' => ['icon' => 'piggy-bank', 'name' => 'Retirement Plan'],
        'pto' => ['icon' => 'umbrella-beach', 'name' => 'Paid Time Off'],
        'education' => ['icon' => 'graduation-cap', 'name' => 'Education Support'],
        'gym' => ['icon' => 'dumbbell', 'name' => 'Gym Membership'],
        'remote_work' => ['icon' => 'home', 'name' => 'Remote Work']
    ];

    // Validation rules
    protected $rules = [
        'jobDetails.jobTitle' => 'required|string|max:255',
        'jobDetails.location' => 'required|string|max:255',
        'jobDetails.jobType' => 'required|string',
        'jobDetails.category' => 'required|string',
        'jobDetails.description' => 'required|string|min:50',

        'salaryBenefits.salaryType' => 'nullable|string',
        'salaryBenefits.currency' => 'nullable|string',
        'salaryBenefits.minSalary' => 'nullable|numeric|required_with:salaryBenefits.maxSalary',
        'salaryBenefits.maxSalary' => 'nullable|numeric|gte:salaryBenefits.minSalary|required_with:salaryBenefits.minSalary',

        'companyInfo.companyWebsite' => 'nullable|url',
        'companyInfo.companySize' => 'required|string',
        'companyInfo.industry' => 'nullable|string',
        'requirements.experienceLevel' => 'nullable|string',
        'requirements.educationLevel' => 'nullable|string',
        'requirements.skills' => 'nullable|string',

        'questionsTab.questions.*.question' => 'required|string',
        'questionsTab.questions.*.type' => 'required|in:multiple_choice,text,checkbox,dropdown',
        'questionsTab.questions.*.is_required' => 'boolean',
        'questionsTab.questions.*.answers.*' => 'required_if:questionsTab.questions.*.type,multiple_choice,checkbox,dropdown|string',
    ];

    // Custom error messages
    protected $messages = [
        'jobDetails.jobTitle.required' => 'Job title is required',
        'jobDetails.location.required' => 'Location is required',
        'jobDetails.jobType.required' => 'Job type is required',
        'jobDetails.category.required' => 'Category is required',
        'jobDetails.description.required' => 'Job description is required',
        'jobDetails.description.min' => 'Job description must be at least 50 characters',
        'salaryBenefits.maxSalary.gte' => 'Maximum salary must be greater than or equal to minimum salary',
    ];

      public $questionsTab = [
        'questions' => []
    ];

    public $questionTypes = [
        'multiple_choice' => 'Multiple Choice',
        'text' => 'Text Answer',
        'checkbox' => 'Checkbox',
        'dropdown' => 'Dropdown'
    ];

    // ... existing methods ...



    // ... existing mount method ...

    public function addQuestion()
    {
        $this->questionsTab['questions'][] = [
            'question' => '',
            'type' => 'multiple_choice',
            'is_required' => false,
            'answers' => ['']
        ];
    }

    // Mount method to check if we're editing an existing job
    public function mount($id = null)
    {
         if ($id) {
            $this->jobId = $id;
            $this->loadJobData();
            $user=Auth::user();
            $businessData=Businessprofile::where('user_id',$user->id)->first();
            $this->businessname=$businessData->businessname;
        }
    }

    public function generateDescription(){
         $title = $this->jobDetails['jobTitle'];
          $aiService = new AIServices();
        $this->jobDetails['description'] = $aiService->loadjobDescription($title);
    }

    public function generateskill(){
        $jbtitle=$this->jobDetails['jobTitle'];
           $aiService = new AIServices();
        $this->requirements['skills']=$aiService->loadSkills($jbtitle);
    }

    // Load job data if editing
    protected function loadJobData()
    {
        $job = JobPost::where('id', $this->jobId)
                     ->where('user_id', Auth::id())
                     ->firstOrFail();

        $this->jobDetails = [
            'jobTitle' => $job->job_title,
            'location' => $job->location,
            'workplaceOption' => $job->workplace_option,
            'jobType' => $job->job_type,
            'category' => $job->category,
            'description' => $job->description
        ];

        $this->salaryBenefits = [
            'salaryType' => $job->salary_type,
            'currency' => $job->currency,
            'minSalary' => $job->min_salary,
            'maxSalary' => $job->max_salary,
            'salaryPeriod' => $job->salary_period,
            'benefits' => $job->benefits ?? []
        ];

        $this->companyInfo = [

            'companyWebsite' => $job->company_website,

            'companySize' => $job->company_size,
            'industry' => $job->industry
        ];

        $this->requirements = [
            'experienceLevel' => $job->experience_level,
            'educationLevel' => $job->education_level,
            'skills' => $job->skills,
            'coverLetter' => $job->require_cover_letter,
            'portfolio' => $job->require_portfolio,
            'references' => $job->require_references
        ];

        $this->previewNotes = $job->preview_notes;

        $this->loadQuestionsData();
    }

     public function removeQuestion($index)
    {
        if (isset($this->questionsTab['questions'][$index]['id'])) {
            // If editing an existing question, mark for deletion
            $this->questionsTab['questions'][$index]['deleted'] = true;
        } else {
            unset($this->questionsTab['questions'][$index]);
        }
        $this->questionsTab['questions'] = array_values($this->questionsTab['questions']);
    }

    public function addAnswer($questionIndex)
    {
        $this->questionsTab['questions'][$questionIndex]['answers'][] = '';
    }

     public function removeAnswer($questionIndex, $answerIndex)
    {
        if (isset($this->questionsTab['questions'][$questionIndex]['answers'][$answerIndex]['id'])) {
            // If editing an existing answer, mark for deletion
            $this->questionsTab['questions'][$questionIndex]['answers'][$answerIndex]['deleted'] = true;
        } else {
            unset($this->questionsTab['questions'][$questionIndex]['answers'][$answerIndex]);
        }
        $this->questionsTab['questions'][$questionIndex]['answers'] = array_values(
            $this->questionsTab['questions'][$questionIndex]['answers']
        );
    }

    // Load questions data if editing
    protected function loadQuestionsData()
    {
        if ($this->jobId) {
            $questions = JobQuestion::with('answers')
                ->where('job_post_id', $this->jobId)
                ->orderBy('order')
                ->get()
                ->toArray();

            $this->questionsTab['questions'] = $questions;
        }
    }

    // Navigation between tabs
    public function navigateTo($tab)
    {
        $this->currentTab = $tab;
    }

    // Save current tab and move to next
    public function saveAndContinue()
    {
        // Validate based on current tab
        if ($this->currentTab === 'job-details') {
            $this->validate([
                'jobDetails.jobTitle' => 'required|string|max:255',
                'jobDetails.location' => 'required|string|max:255',
                'jobDetails.jobType' => 'required|string',
                'jobDetails.category' => 'required|string',
                'jobDetails.description' => 'required|string|min:50',
            ]);

            // Save job details
            $this->saveJobDetails();

            $this->currentTab = 'salary-benefits';

        } elseif ($this->currentTab === 'salary-benefits') {
              $this->validate([
                'salaryBenefits.salaryType' => 'required',
                'salaryBenefits.currency' => 'required',

            ]);
            // Only validate if salary type is range
            if ($this->salaryBenefits['salaryType'] === 'range') {

                $this->validate([
                    'salaryBenefits.minSalary' => 'required|numeric',
                    'salaryBenefits.maxSalary' => 'required|numeric|gte:salaryBenefits.minSalary',
                ]);
            }
            else{
                 $this->saveSalaryBenefits();
            }

            // Save salary benefits


            $this->currentTab = 'company-info';

        } elseif ($this->currentTab === 'company-info') {
            $this->validate([
                'companyInfo.companySize' => 'required',
                'companyInfo.companyWebsite' => 'nullable|url',
            ]);

            // Save company info
            $this->saveCompanyInfo();

            $this->currentTab = 'requirements';

        } elseif ($this->currentTab === 'requirements') {
            // Save requirements
            $this->validate([
                  'requirements.experienceLevel' => 'required',
                   'requirements.educationLevel' => 'required',
            ]);
            $this->saveRequirements();

            $this->currentTab = 'questions'; // Changed from 'preview' to 'questions'

        } elseif ($this->currentTab === 'questions') {
            // Save questions
            $this->saveQuestions();
            $this->currentTab = 'preview';

        } elseif ($this->currentTab === 'preview') {
            // Final validation before submission
            $this->validate();

            // Save the complete job post
            $this->saveJobPost();

            // Show success message
            session()->flash('message', 'Job post created successfully!');

            // Redirect to job list or show success message
            // return redirect()->route('employer.jobs');
        }
    }

      protected function saveQuestions()
    {
        if (!$this->jobId) return;

        foreach ($this->questionsTab['questions'] as $index => $questionData) {
            // Skip deleted questions
            if (isset($questionData['deleted']) && $questionData['deleted']) {
                if (isset($questionData['id'])) {
                    JobQuestion::where('id', $questionData['id'])->delete();
                }
                continue;
            }

            $question = JobQuestion::updateOrCreate(
                ['id' => $questionData['id'] ?? null],
                [
                    'job_post_id' => $this->jobId,
                    'question' => $questionData['question'],
                    'type' => $questionData['type'],
                    'is_required' => $questionData['is_required'] ?? false,
                    'order' => $index
                ]
            );

            // Save answers for question types that need them
            if (in_array($questionData['type'], ['multiple_choice', 'checkbox', 'dropdown']) &&
                isset($questionData['answers'])) {

                foreach ($questionData['answers'] as $answerIndex => $answerData) {
                    // Skip deleted answers
                    if (isset($answerData['deleted']) && $answerData['deleted']) {
                        if (isset($answerData['id'])) {
                            JobQuestionAnswer::where('id', $answerData['id'])->delete();
                        }
                        continue;
                    }

                    JobQuestionAnswer::updateOrCreate(
                        ['id' => $answerData['id'] ?? null],
                        [
                            'job_question_id' => $question->id,
                            'answer' => $answerData,
                            'order' => $answerIndex
                        ]
                    );
                }
            }
        }
    }
    // Save job details to database
    protected function saveJobDetails()
    {
        $data = [
            'job_title' => $this->jobDetails['jobTitle'],
            'location' => $this->jobDetails['location'],
            'workplace_option' => $this->jobDetails['workplaceOption'],
            'job_type' => $this->jobDetails['jobType'],
            'category' => $this->jobDetails['category'],
            'description' => $this->jobDetails['description'],
            'status' => 'draft'
        ];

        if ($this->jobId) {
            JobPost::where('id', $this->jobId)->update($data);
        } else {
            $data['user_id'] = Auth::id();
            $job = JobPost::create($data);
            $this->jobId = $job->id;
        }
    }

    // Save salary benefits to database
    protected function saveSalaryBenefits()
    {
        if ($this->jobId) {
           JobPost::where('id', $this->jobId)->update([
    'salary_type' => $this->salaryBenefits['salaryType'],
    'currency' => $this->salaryBenefits['currency'],
    'min_salary' => $this->salaryBenefits['salaryType'] !== 'range' ? 0 : ($this->salaryBenefits['minSalary'] ?? 0),
    'max_salary' => $this->salaryBenefits['salaryType'] !== 'range' ? 0 : ($this->salaryBenefits['maxSalary'] ?? 0),
    'salary_period' => $this->salaryBenefits['salaryPeriod'],
    'benefits' => $this->salaryBenefits['benefits']
]);
        }
    }

    // Save company info to database
    protected function saveCompanyInfo()
    {
        if ($this->jobId) {
            JobPost::where('id', $this->jobId)->update([
                'company_name' => $this->businessname,
                'company_website' => $this->companyInfo['companyWebsite'],
                'company_size' => $this->companyInfo['companySize'],
                'industry' => $this->companyInfo['industry']
            ]);
        }
    }

    // Save requirements to database
    protected function saveRequirements()
    {
        if ($this->jobId) {

            JobPost::where('id', $this->jobId)->update([
                'experience_level' => $this->requirements['experienceLevel'],
                'education_level' => $this->requirements['educationLevel'],
                'skills' => $this->requirements['skills'],
                'require_cover_letter' => $this->requirements['coverLetter'],
                'require_portfolio' => $this->requirements['portfolio'],
                'require_references' => $this->requirements['references']
            ]);
        }
    }

    // Save the complete job post
    protected function saveJobPost()
    {
        sleep(5);
    $this->dispatch('jobSaved');
        if ($this->jobId) {
            JobPost::where('id', $this->jobId)->update([
                'preview_notes' => $this->previewNotes,
                'status' => 'published'
            ]);
        }
    }

    // Go back to previous tab
    public function goBack()
    {
        $tabs = ['job-details', 'salary-benefits', 'company-info', 'requirements', 'questions', 'preview'];
        $currentIndex = array_search($this->currentTab, $tabs);

        if ($currentIndex > 0) {
            $this->currentTab = $tabs[$currentIndex - 1];
        }
    }

    // Save as draft
    public function saveDraft()
    {
        // Save all current data to draft
        if ($this->jobId) {
            JobPost::where('id', $this->jobId)->update(['status' => 'draft']);
            session()->flash('message', 'Draft saved successfully!');
        } else {
            // Create a new draft if no job ID exists
            $this->saveJobDetails();
            session()->flash('message', 'Draft saved successfully!');
        }
    }

    // Toggle workplace option
    public function selectWorkplaceOption($option)
    {
        $this->jobDetails['workplaceOption'] = $option;
    }

    // Toggle benefit selection
    public function toggleBenefit($benefitKey)
    {
        if (in_array($benefitKey, $this->salaryBenefits['benefits'])) {
            $this->salaryBenefits['benefits'] = array_diff($this->salaryBenefits['benefits'], [$benefitKey]);
        } else {
            $this->salaryBenefits['benefits'][] = $benefitKey;
        }

        // Save benefits immediately when toggled
        if ($this->jobId) {
            JobPost::where('id', $this->jobId)->update([
                'benefits' => $this->salaryBenefits['benefits']
            ]);
        }
    }

    // Reset the form
    protected function resetForm()
    {
        $this->reset([
            'jobDetails',
            'salaryBenefits',
            'companyInfo',
            'requirements',
            'previewNotes'
        ]);

        $this->currentTab = 'job-details';
        $this->salaryBenefits['benefits'] = [];
        $this->jobDetails['workplaceOption'] = 'on-site';
        $this->jobId = null;
    }

    public function render()
    {
        return view('livewire.web.index.employer.job.createjob');
    }
}
