<?php

namespace App\Livewire\Admin\Dashboard\Jobcategory;

use App\Models\JobCategory;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;

class Jobcategorydashboard extends Component
{

   use WithFileUploads;

    public $categories;
    public $name;
    public $description,$base;
    public $image;
    public $status = true;
    public $editingId = null;

    protected $rules = [
        'name' => 'required|string|max:255',
        'description' => 'nullable|string',
        'image' => 'nullable|image|max:2048',
        'base'=>'required',
        'status' => 'boolean'
    ];

    public function mount()
    {
        $this->loadCategories();
    }

    public function loadCategories()
    {
        $this->categories = JobCategory::all();
    }

    public function saveCategory()
    {
        $this->validate();

        $imageName = null;
        if ($this->image) {
            // Ensure directory exists
               $extension = $this->image->getClientOriginalExtension();
            $uniqueFilename = 'job_category_' . time() . '_' . uniqid() . '.' . $extension;

            // Store image using storeAs
            $this->image->storeAs('job-categories', $uniqueFilename, 'public');
            $imageName = $uniqueFilename;
        }

        if ($this->editingId) {
            // Update existing category
            $category = JobCategory::findOrFail($this->editingId);

            // Delete old image if new one is uploaded
            if ($this->image && $category->image) {
                Storage::disk('public')->delete('job-categories/' . $category->image);
            }

            $category->update([
                'name' => $this->name,
                'description' => $this->description,
                'base'=>$this->base,
                'image' => $imageName ?: $category->image,
                'status' => $this->status
            ]);

            session()->flash('message', 'Category updated successfully!');
        } else {
            // Create new category
            JobCategory::create([
                'name' => $this->name,
                'description' => $this->description,
                'base'=>$this->base,
                'image' => $imageName,
                'status' => $this->status
            ]);

            session()->flash('message', 'Category created successfully!');
        }

        $this->resetForm();
        $this->loadCategories();
    }

    public function editCategory($id)
    {
        $category = JobCategory::findOrFail($id);
        $this->editingId = $category->id;
        $this->name = $category->name;
        $this->description = $category->description;
        $this->base=$category->base;
        $this->status = $category->status;
        $this->image = null;
    }

    public function deleteCategory($id)
    {
        $category = JobCategory::findOrFail($id);

        // Delete image if exists
        if ($category->image) {
            Storage::disk('public')->delete('job-categories/' . $category->image);
        }

        $category->delete();

        session()->flash('message', 'Category deleted successfully!');
        $this->loadCategories();
    }

    public function resetForm()
    {
        $this->reset(['name', 'description', 'image', 'status', 'editingId']);
        $this->resetValidation();
    }

    public function render()
    {
        return view('livewire.admin.dashboard.jobcategory.jobcategorydashboard')
            ->layout('livewire.admin.dashboard.layout.adminlayout');
    }
}
